<?php
// busqueda.php - Backend para la consulta externa

header('Content-Type: application/json'); // Indicar que la respuesta será JSON

// Incluir configuración de la base de datos (ajusta la ruta si es necesario)
// Asume que db.php define la variable $pdo
require_once 'config/db.php'; // Si busqueda.php está en la raíz

// Verificar si se recibió el término de búsqueda y si pdo está disponible
if (!isset($pdo)) {
    echo json_encode(['status' => 'error', 'message' => 'Error interno del servidor: No se pudo conectar a la base de datos.']);
    exit;
}
if (!isset($_POST['term']) || empty(trim($_POST['term']))) {
    echo json_encode(['status' => 'error', 'message' => 'No se proporcionó término de búsqueda.']);
    exit;
}

$term = trim($_POST['term']);
$results = []; // Array para almacenar los resultados formateados

try {
    // Preparar la consulta principal
    // Buscar por Placa, Tablero (en vehiculos) o DNI (en propietarios)
    // Se une siempre con propietario y opcionalmente con conductor
    $sql = "SELECT
                v.id as vehiculo_id, v.placa, v.color, v.marca, v.modelo, v.anio_modelo, v.tablero, v.anotacion1, v.anotacion2,
                p.id as propietario_id, p.nombre as p_nombre, p.apellido as p_apellido, p.dni as p_dni, p.direccion as p_direccion, p.provincia as p_provincia, p.distrito as p_distrito, p.telefono as p_telefono,
                c.id as conductor_id, c.nombre as c_nombre, c.apellido as c_apellido, c.dni as c_dni, c.licencia_conducir as c_licencia, c.telefono as c_telefono
            FROM vehiculos v
            JOIN propietarios p ON v.propietario_id = p.id
            LEFT JOIN conductores c ON v.conductor_id = c.id
            WHERE v.placa = :term_placa
               OR v.tablero LIKE :term_tablero
               OR p.dni = :term_dni";

    $stmt = $pdo->prepare($sql);

    // Preparar términos para la consulta (ajusta LIKE si quieres búsquedas parciales en tablero)
    $termPlaca = $term;
    $termTablero = "%" . $term . "%"; // Búsqueda parcial para tablero
    $termDni = $term;

    $stmt->bindParam(':term_placa', $termPlaca, PDO::PARAM_STR);
    $stmt->bindParam(':term_tablero', $termTablero, PDO::PARAM_STR);
    $stmt->bindParam(':term_dni', $termDni, PDO::PARAM_STR);

    $stmt->execute();
    $rawData = $stmt->fetchAll(PDO::FETCH_ASSOC);

    // Formatear los resultados para una estructura JSON más limpia
    if ($rawData) {
        foreach ($rawData as $row) {
            $results[] = [
                'vehiculo' => [
                    'id' => $row['vehiculo_id'],
                    'placa' => $row['placa'],
                    'color' => $row['color'],
                    'marca' => $row['marca'],
                    'modelo' => $row['modelo'],
                    'anio_modelo' => $row['anio_modelo'],
                    'tablero' => $row['tablero'],
                    'anotacion1' => $row['anotacion1'],
                    'anotacion2' => $row['anotacion2'],
                ],
                'propietario' => [
                    'id' => $row['propietario_id'],
                    'nombre' => $row['p_nombre'],
                    'apellido' => $row['p_apellido'],
                    'dni' => $row['p_dni'],
                    'direccion' => $row['p_direccion'],
                    'provincia' => $row['p_provincia'],
                    'distrito' => $row['p_distrito'],
                    'telefono' => $row['p_telefono'],
                ],
                // Incluir conductor solo si existe (conductor_id no es NULL)
                'conductor' => $row['conductor_id'] ? [
                    'id' => $row['conductor_id'],
                    'nombre' => $row['c_nombre'],
                    'apellido' => $row['c_apellido'],
                    'dni' => $row['c_dni'],
                    'licencia_conducir' => $row['c_licencia'],
                    'telefono' => $row['c_telefono'],
                ] : null // Si no hay conductor, se envía null
            ];
        }
        // Devolver el array de resultados formateados
        echo json_encode($results);

    } else {
        // No se encontraron resultados
        echo json_encode(['status' => 'not_found', 'message' => 'No se encontraron vehículos o propietarios que coincidan con el término de búsqueda.']);
    }

} catch (PDOException $e) {
    // Error de base de datos
    error_log("Error en busqueda.php: " . $e->getMessage()); // Registrar error real
    echo json_encode(['status' => 'error', 'message' => 'Error al consultar la base de datos. Inténtelo más tarde.']);
}

exit; // Terminar script
?>