<?php
// index_busqueda.php - Búsqueda del Lado del Servidor (Banner Superior - Revisado)

$resultados = [];
$error_busqueda = null;
$termino_buscado = isset($_GET['term']) ? trim($_GET['term']) : '';

// Función para escapar HTML
function escapeHtml($unsafe) {
    if ($unsafe === null || $unsafe === '') return '-';
    return htmlspecialchars(strval($unsafe), ENT_QUOTES, 'UTF-8');
}

// --- Lógica de Búsqueda ---
if (!empty($termino_buscado)) {
    // ASUME que este archivo está en la RAÍZ y config/ está dentro
    require_once 'config/db.php';
    if (!isset($pdo)) {
        $error_busqueda = "Error interno: No se pudo conectar a la BD.";
    } else {
        try {
            $sql = "SELECT
                        v.id as vehiculo_id, v.placa, v.color, v.marca, v.modelo, v.anio_modelo, v.tablero, v.anotacion1, v.anotacion2,
                        p.id as propietario_id, p.nombre as p_nombre, p.apellido as p_apellido, p.dni as p_dni, p.direccion as p_direccion, p.provincia as p_provincia, p.distrito as p_distrito, p.telefono as p_telefono,
                        c.id as conductor_id, c.nombre as c_nombre, c.apellido as c_apellido, c.dni as c_dni, c.licencia_conducir as c_licencia, c.telefono as c_telefono
                    FROM vehiculos v
                    JOIN propietarios p ON v.propietario_id = p.id
                    LEFT JOIN conductores c ON v.conductor_id = c.id
                    WHERE v.placa = :term_placa
                       OR v.tablero LIKE :term_tablero
                       OR p.dni = :term_dni";

            $stmt = $pdo->prepare($sql);
            $termPlaca = $termino_buscado;
            $termTablero = "%" . $termino_buscado . "%";
            $termDni = $termino_buscado;
            $stmt->bindParam(':term_placa', $termPlaca, PDO::PARAM_STR);
            $stmt->bindParam(':term_tablero', $termTablero, PDO::PARAM_STR);
            $stmt->bindParam(':term_dni', $termDni, PDO::PARAM_STR);
            $stmt->execute();
            $resultados = $stmt->fetchAll(PDO::FETCH_ASSOC);

        } catch (PDOException $e) {
            error_log("Error en index_busqueda.php: " . $e->getMessage());
            $error_busqueda = "Error al consultar la base de datos.";
        }
    }
}

// --- Lógica del Tema Oscuro (Leído desde Cookie) ---
// Usar 'search_theme' como nombre de cookie específico
$themeClass = (isset($_COOKIE['search_theme']) && $_COOKIE['search_theme'] === 'dark') ? 'dark-mode' : '';
$themeIcon = ($themeClass === 'dark-mode') ? 'fa-sun' : 'fa-moon';
$themeText = ($themeClass === 'dark-mode') ? ' Modo Claro' : ' Modo Oscuro';

?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Consulta Vehicular - SysVehiculos</title>
    <!-- Favicon principal para la mayoría de navegadores -->
    <link rel="icon" href="img/favicon.ico" sizes="any"> <!-- Busca .ico en la raíz -->
    <!-- Alternativa SVG (si tienes uno) -->
    <!-- <link rel="icon" href="/img/logo.svg" type="image/svg+xml"> -->
    <!-- Icono para dispositivos Apple (180x180 recomendado) -->
    <link rel="apple-touch-icon" href="/img/apple-touch-icon.png"> <!-- Asume que está en /img/ -->
    <!-- Opcional: Manifiesto para PWA/Android -->
    <!-- <link rel="manifest" href="/manifest.webmanifest"> -->
    <!-- CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <!-- Asegúrate que esta ruta es correcta desde donde está index_busqueda.php -->
    <link rel="stylesheet" href="css/style_busqueda.css">
</head>
<body class="search-page-body <?php echo $themeClass; ?>"> <!-- Aplicar clase de tema desde PHP -->

    <div class="page-wrapper">

        <!-- Banner Section -->
        <div class="banner-container mb-4 main-content-width">
             <!-- Asegúrate que esta ruta es correcta -->
            <img src="img/banner_consulta.jpg" alt="Banner Consulta Vehicular" class="img-fluid banner-image">
        </div>

        <!-- Contenedor principal -->
        <div class="container search-container my-4 position-relative main-content-width">

            <!-- Botón Tema -->
            <div class="theme-toggle-container position-absolute top-0 end-0 p-2 pt-3 pe-3">
                 <button class="btn btn-outline-secondary btn-sm theme-toggle-btn" id="themeToggleBtn" title="Cambiar tema">
                    <i class="fas <?php echo $themeIcon; ?>"></i> <span class="d-none d-md-inline" id="themeToggleText"><?php echo $themeText; ?></span>
                </button>
            </div>

            <!-- Título Principal -->
            <h2 class="text-center mb-4 page-title">Consulta Vehicular</h2>

            <!-- Search Form -->
            <form id="searchForm" class="mb-4" action="index_busqueda.php" method="GET">
                <div class="input-group input-group-lg shadow-sm">
                    <input type="search" id="searchInput" name="term" class="form-control" placeholder="Ingrese DNI, Placa o Tablero..." aria-label="Término de búsqueda" value="<?php echo htmlspecialchars($termino_buscado); ?>" required>
                    <button class="btn btn-primary" type="submit" id="searchButton">
                        <i class="fas fa-search"></i> <span class="d-none d-sm-inline">Buscar</span>
                    </button>
                </div>
            </form>

            <!-- Results Area -->
            <div id="searchResults" class="results-area mt-4" <?php echo empty($termino_buscado) ? 'style="display: none;"' : ''; ?>>
                 <?php if (!empty($termino_buscado)): ?>
                    <?php if ($error_busqueda): ?>
                        <div class="alert alert-danger error-message"><?php echo escapeHtml($error_busqueda); ?></div>
                    <?php elseif (empty($resultados)): ?>
                        <div class="alert alert-warning no-results-message">No se encontraron resultados para "<?php echo escapeHtml($termino_buscado); ?>".</div>
                    <?php else: ?>
                        <?php foreach ($resultados as $index => $row): ?>
                            <div class="result-item mb-4 <?php echo $index > 0 ? 'mt-4 pt-4 border-top' : ''; ?>">
                                <div class="row g-3">
                                    <!-- Columna Vehículo -->
                                    <div class="col-md-6 col-lg-4">
                                        <div class="card h-100 shadow-sm result-card">
                                            <div class="card-header card-header-icon"> <i class="fas fa-car me-2"></i> Vehículo: <strong><?php echo escapeHtml($row['placa']); ?></strong> </div>
                                            <div class="card-body"> <dl class="data-list mb-0"> <dt>Marca:</dt><dd><?php echo escapeHtml($row['marca']); ?></dd> <dt>Modelo:</dt><dd><?php echo escapeHtml($row['modelo']); ?></dd> <dt>Año:</dt><dd><?php echo escapeHtml($row['anio_modelo']); ?></dd> <dt>Color:</dt><dd><?php echo escapeHtml($row['color']); ?></dd> <dt>Tablero:</dt><dd><?php echo escapeHtml($row['tablero']); ?></dd> <dt>Anotación 1:</dt><dd><?php echo escapeHtml($row['anotacion1']); ?></dd> <dt>Anotación 2:</dt><dd><?php echo escapeHtml($row['anotacion2']); ?></dd> </dl> </div>
                                        </div>
                                    </div>
                                    <!-- Columna Propietario -->
                                    <div class="col-md-6 col-lg-4">
                                        <div class="card h-100 shadow-sm result-card">
                                            <div class="card-header card-header-icon"> <i class="fas fa-user-tie me-2"></i> Propietario </div>
                                            <div class="card-body"> <?php if ($row['propietario_id']): ?> <dl class="data-list mb-0"> <dt>Nombre:</dt><dd><?php echo escapeHtml($row['p_nombre']) . ' ' . escapeHtml($row['p_apellido']); ?></dd> <dt>DNI:</dt><dd><?php echo escapeHtml($row['p_dni']); ?></dd> <dt>Dirección:</dt><dd><?php echo escapeHtml($row['p_direccion']) . ', ' . escapeHtml($row['p_distrito']) . ', ' . escapeHtml($row['p_provincia']); ?></dd> <dt>Teléfono:</dt><dd><?php echo escapeHtml($row['p_telefono']); ?></dd> </dl> <?php else: ?> <p class="text-muted mb-0"><em>Datos del propietario no disponibles.</em></p> <?php endif; ?> </div>
                                        </div>
                                    </div>
                                    <!-- Columna Conductor -->
                                    <div class="col-md-12 col-lg-4">
                                         <div class="card h-100 shadow-sm result-card">
                                            <div class="card-header card-header-icon"> <i class="fas fa-id-card me-2"></i> Conductor Asignado </div>
                                            <div class="card-body"> <?php if ($row['conductor_id']): ?> <dl class="data-list mb-0"> <dt>Nombre:</dt><dd><?php echo escapeHtml($row['c_nombre']) . ' ' . escapeHtml($row['c_apellido']); ?></dd> <dt>DNI:</dt><dd><?php echo escapeHtml($row['c_dni']); ?></dd> <dt>Licencia:</dt><dd><?php echo escapeHtml($row['c_licencia']); ?></dd> <dt>Teléfono:</dt><dd><?php echo escapeHtml($row['c_telefono']); ?></dd> </dl> <?php else: ?> <p class="text-muted mb-0"><em>Vehículo sin conductor asignado en el sistema.</em></p> <?php endif; ?> </div>
                                        </div>
                                    </div>
                                </div> <!-- /row -->
                            </div><!-- /result-item -->
                        <?php endforeach; ?>
                    <?php endif; ?>
                <?php endif; ?>
            </div> <!-- /searchResults -->

        </div><!-- /container -->
    </div> <!-- /page-wrapper -->

    <!-- Footer -->
    <footer class="text-center mt-auto py-3 footer-area main-content-width">
        <p class="mb-1 small">© <?php echo date("Y"); ?> SysVehiculos. Todos los derechos reservados.</p>
        <p class="mb-0 small">Desarrollado por: Ing. Felipe Altamirano Carrasco</p>
    </footer>

    <!-- Scripts -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <!-- Ya NO se necesita jQuery ni script_busqueda.js complejo -->
    <script>
        // --- Script MÍNIMO solo para el botón de Tema Oscuro ---
        document.addEventListener('DOMContentLoaded', function() {
            const themeToggleBtn = document.getElementById('themeToggleBtn');
            const themeToggleText = document.getElementById('themeToggleText');
            const body = document.body;

            themeToggleBtn?.addEventListener('click', () => {
                const isDarkMode = body.classList.toggle('dark-mode');
                const newTheme = isDarkMode ? 'dark' : 'light';
                const newIcon = isDarkMode ? 'fa-sun' : 'fa-moon';
                const newText = isDarkMode ? ' Modo Claro' : ' Modo Oscuro';
                // Guardar preferencia en Cookie con nombre específico
                document.cookie = `search_theme=${newTheme};path=/;max-age=31536000;SameSite=Lax`;
                const iconElement = themeToggleBtn.querySelector('i');
                if (iconElement) iconElement.className = `fas ${newIcon}`;
                if (themeToggleText) themeToggleText.textContent = newText;
                 console.log(`Tema cambiado a: ${newTheme}`);
            });
        });
    </script>
</body>
</html>