<?php
// --- procesar_login.php ---
// Objetivo: Validar las credenciales del usuario contra la base de datos e iniciar sesión.

// 1. Iniciar/Reanudar la sesión ANTES de cualquier salida.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// 2. Configuración de la Base de Datos (Asegúrate que estos datos sean correctos)
// (Alternativa recomendada: Incluir require_once 'config/db.php'; en lugar de definir aquí)
define('DB_HOST', 'localhost');        // Servidor de la BD (normalmente localhost)
define('DB_NAME', 'sisvehiculos');     // Nombre EXACTO de tu base de datos
define('DB_USER', 'kasparov');         // Usuario EXACTO de la BD
define('DB_PASS', 'nicofeli');         // Contraseña EXACTA del usuario de la BD
define('DB_CHARSET', 'utf8mb4');       // Juego de caracteres (recomendado)

// 3. Verificar si el script fue llamado mediante POST (desde el formulario)
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // 4. Validar que los campos no estén vacíos (después de quitar espacios)
    $username_or_email = isset($_POST["username"]) ? trim($_POST["username"]) : '';
    $password = isset($_POST["password"]) ? trim($_POST["password"]) : '';

    if (empty($username_or_email) || empty($password)) {
        // Error: Campos vacíos
        $_SESSION['login_error'] = "Por favor, ingresa tu usuario/email y contraseña.";
        header("Location: index.php?error=empty_fields"); // Redirigir de vuelta al login
        exit; // Detener script
    }

    // 5. Intentar conexión a la Base de Datos usando PDO
    $dsn = "mysql:host=" . DB_HOST . ";dbname=" . DB_NAME . ";charset=" . DB_CHARSET;
    $options = [
        PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION, // Lanzar excepciones en errores SQL
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,       // Devolver resultados como array asociativo
        PDO::ATTR_EMULATE_PREPARES   => false,                  // Usar preparaciones nativas (más seguro)
    ];

    try {
        // Crear instancia de PDO para la conexión
        $pdo = new PDO($dsn, DB_USER, DB_PASS, $options);

        // 6. Preparar la consulta SQL (CON 'rol' AÑADIDO Y PLACEHOLDERS DIFERENTES)
        $sql = "SELECT id, username, password_hash, activo, rol -- << ROL AÑADIDO AQUÍ
                FROM usuarios
                WHERE username = :username_check OR email = :email_check
                LIMIT 1";
        $stmt = $pdo->prepare($sql);

        // 7. Vincular el valor de entrada a AMBOS placeholders
        $stmt->bindParam(':username_check', $username_or_email, PDO::PARAM_STR);
        $stmt->bindParam(':email_check', $username_or_email, PDO::PARAM_STR);

        // 8. Ejecutar la consulta
        $stmt->execute();

        // 9. Obtener el resultado (si existe)
        $user = $stmt->fetch(); // Devuelve el usuario o false si no se encontró

        // 10. Verificar si se encontró un usuario y si la contraseña y el estado son correctos
        if ($user && $user['activo'] == 1 && password_verify($password, $user['password_hash'])) {
            // ¡Login Exitoso! Usuario encontrado, activo y contraseña coincide.

            // 11. Regenerar el ID de sesión para seguridad (previene fijación de sesión)
            session_regenerate_id(true);

            // 12. Limpiar cualquier error de login anterior
            unset($_SESSION['login_error']);

            // 13. Guardar datos esenciales del usuario en la sesión (INCLUYENDO EL ROL)
            // ¡NUNCA guardes la contraseña o su hash en la sesión!
            $_SESSION['loggedin'] = true;
            $_SESSION['user_id'] = $user['id'];
            $_SESSION['username'] = $user['username'];
            $_SESSION['user_role'] = $user['rol']; // <<<--- ROL GUARDADO AQUÍ ---<<<
            // Puedes añadir más datos si los necesitas globalmente y los seleccionaste
            // $_SESSION['nombre_completo'] = $user['nombre_completo'];

            // 14. (Opcional pero recomendado) Actualizar el campo 'ultimo_login'
            try {
                $sqlUpdate = "UPDATE usuarios SET ultimo_login = NOW() WHERE id = :id";
                $stmtUpdate = $pdo->prepare($sqlUpdate);
                // Usar el ID obtenido del usuario que acaba de iniciar sesión
                $stmtUpdate->bindParam(':id', $user['id'], PDO::PARAM_INT);
                $stmtUpdate->execute();
            } catch (PDOException $e) {
                // Si falla la actualización del último login, no es crítico para el flujo principal.
                // Solo registramos el error para revisión posterior.
                error_log("SysVehiculos - Error al actualizar ultimo_login para usuario ID {$user['id']}: " . $e->getMessage());
            }

            // 15. Redirigir al panel principal (dashboard)
            header("Location: dashboard.php");
            exit; // Detener script después de la redirección

        } else {
            // Error: Credenciales incorrectas o usuario inactivo/no encontrado
            // Damos un mensaje genérico por seguridad (no revelar si falló usuario o contraseña)
            $_SESSION['login_error'] = "Usuario o contraseña incorrectos, o cuenta inactiva.";
            header("Location: index.php?error=invalid_credentials"); // Redirigir de vuelta al login
            exit; // Detener script
        }

    } catch (PDOException $e) {
        // ¡Error Crítico! Falló la conexión a la BD o la consulta SQL.

        // 16. Registrar el error DETALLADO en el log del servidor (para el desarrollador)
        error_log("SysVehiculos - Error de Base de Datos en Login: Code[{$e->getCode()}] - " . $e->getMessage());

        // 17. Establecer mensaje de error GENÉRICO para el usuario (no exponer detalles)
        $_SESSION['login_error'] = "Error del sistema. Por favor, inténtalo más tarde.";
        header("Location: index.php?error=db_error"); // Redirigir de vuelta al login
        exit; // Detener script
    }

} else {
    // Si alguien intenta acceder a este archivo directamente sin usar el método POST
    $_SESSION['login_error'] = "Acceso no permitido.";
    header("Location: index.php"); // Redirigir a la página de inicio
    exit; // Detener script
}
?>