<?php
// *** PÁGINA PÚBLICA - Requiere su propia conexión ***
require_once 'config.php';
require_once 'includes/database.php';

$mensaje_error = '';
$resultados_votos = [];
$total_general_votos = 0;
$chart_labels = [];
$chart_data = [];
$chart_background_colors = [];
$id_eleccion_actual = DEFAULT_ELECTION_ID; // Podrías obtener esto de un parámetro GET si quieres mostrar resultados de diferentes elecciones
$nombre_eleccion = "Elección ID " . $id_eleccion_actual; // Nombre por defecto

try {
    $pdo = Database::connect();

    // OBTENER NOMBRE DE LA ELECCIÓN (Opcional)
    $sql_elec = "SELECT nombre FROM elecciones WHERE id = ? LIMIT 1";
    $q_elec = $pdo->prepare($sql_elec);
    $q_elec->execute([$id_eleccion_actual]);
    $eleccion_info = $q_elec->fetch(PDO::FETCH_ASSOC);
    if ($eleccion_info) {
        $nombre_eleccion = $eleccion_info['nombre'];
    }

    // --- Carga de Datos desde la BD (MISMA LÓGICA que reporte_votos.php) ---
     $sql_resultados = "SELECT
                            c.nombre, c.apellido, c.partido_politico, c.foto_url,
                            SUM(v.cantidad_votos) AS total_votos
                        FROM votos AS v
                        JOIN candidatos AS c ON v.id_candidato = c.id
                        WHERE v.id_eleccion = ?
                        GROUP BY c.id, c.nombre, c.apellido, c.partido_politico, c.foto_url
                        ORDER BY total_votos DESC";

    $q_resultados = $pdo->prepare($sql_resultados);
    $q_resultados->execute([$id_eleccion_actual]);
    $resultados_db = $q_resultados->fetchAll(PDO::FETCH_ASSOC);

     if (empty($resultados_db)) {
        $mensaje_error = "Aún no hay resultados disponibles para esta elección.";
    } else {
        $resultados_votos = $resultados_db;
        // Procesamiento para el gráfico y la tabla
        $colores_base = ['#36a2eb', '#ff6384', '#ffce56', '#4bc0c0', '#9966ff', '#ff9f40'];
        $color_index = 0;
        foreach ($resultados_votos as $resultado) {
             $nombre_completo_chart = $resultado['nombre'] . ' ' . $resultado['apellido'];
            $total_general_votos += $resultado['total_votos'];
            $chart_labels[] = $nombre_completo_chart;
            $chart_data[] = $resultado['total_votos'];
            $chart_background_colors[] = $colores_base[$color_index % count($colores_base)];
            $color_index++;
        }
    }

    Database::disconnect(); // Cerrar conexión

} catch (PDOException $e) {
    error_log("Error en página pública de resultados: " . $e->getMessage());
    $mensaje_error = "No se pudieron cargar los resultados. Intente más tarde.";
    // Asegurarse de desconectar si la conexión se estableció antes del error
    if (isset($pdo)) { Database::disconnect(); }
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Resultados Electorales - <?php echo htmlspecialchars($nombre_eleccion); ?></title>
    <!-- Bootstrap CSS -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Font Awesome -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css">
    <style>
        body { background-color: #f8f9fa; padding-top: 20px; padding-bottom: 40px; }
        .card-header { background-color: #e9ecef; font-weight: bold; }
        .img-thumbnail { background-color: #fff; }
        .chart-legend-item { display: inline-flex; align-items: center; margin: 5px 10px; padding: 5px 8px; border-radius: 5px; color: white; text-shadow: 1px 1px 2px black; }
        .chart-legend-item img { margin-right: 8px; border: 1px solid #ccc; border-radius: 3px; }
        .chart-legend-item small { font-size: 0.9em; font-weight: 500;}
        .footer-info { margin-top: 30px; padding-top: 15px; border-top: 1px solid #dee2e6; font-size: 0.9em; color: #6c757d; }
    </style>
</head>
<body>
<div class="container">
    <header class="text-center mb-4">
        <h1 class="display-5"><i class="fas fa-poll"></i> Resultados Electorales</h1>
        <h2><?php echo htmlspecialchars($nombre_eleccion); ?></h2>
        <p class="lead">Consulta los resultados actualizados.</p>
        <p><small>Última actualización de esta página: <?php echo date('d/m/Y H:i:s'); ?></small></p>
    </header>
    <main>
        <?php if ($mensaje_error): ?>
            <div class="alert alert-warning text-center" role="alert">
                <i class="fas fa-info-circle"></i> <?php echo htmlspecialchars($mensaje_error); ?>
            </div>
        <?php elseif (!empty($resultados_votos)): ?>
            <!-- Gráfico -->
            <div class="card shadow-sm mb-4">
                <div class="card-header">Resultados Gráficos</div>
                <div class="card-body">
                    <div style="height: 400px; position: relative;"><canvas id="resultsChart"></canvas></div>
                    <div class="mt-4 text-center">
                         <?php foreach ($resultados_votos as $index => $resultado): ?>
                            <div class="chart-legend-item" style="background-color: <?php echo $chart_background_colors[$index]; ?>;">
                                 <img src="<?php echo !empty($resultado['foto_url']) ? htmlspecialchars($resultado['foto_url']) : 'assets/placeholder.png'; ?>"
                                     alt="Logo" class="img-thumbnail" width="35" height="35" style="object-fit: contain;"
                                      onerror="this.onerror=null; this.src='assets/placeholder.png';">
                                 <small><?php echo htmlspecialchars($resultado['nombre'] . ' ' . $resultado['apellido']); ?></small>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>
            <!-- Tabla -->
            <div class="card shadow-sm mb-4">
                <div class="card-header">Resultados Detallados</div>
                <div class="card-body">
                    <p class="text-end fw-bold">Total General de Votos Válidos: <?php echo number_format($total_general_votos); ?></p>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover table-striped">
                             <thead class="table-light">
                                <tr>
                                    <th class="text-center">Logo</th><th>Candidato</th><th>Partido</th>
                                    <th class="text-end">Votos</th><th class="text-end" style="min-width: 150px;">Porcentaje (%)</th>
                                </tr>
                            </thead>
                             <tbody>
                                <?php foreach ($resultados_votos as $resultado): ?>
                                    <?php
                                    $porcentaje = ($total_general_votos > 0) ? ($resultado['total_votos'] / $total_general_votos) * 100 : 0;
                                     $nombre_completo = $resultado['nombre'] . ' ' . $resultado['apellido'];
                                    ?>
                                    <tr>
                                        <td class="text-center align-middle">
                                            <img src="<?php echo !empty($resultado['foto_url']) ? htmlspecialchars($resultado['foto_url']) : 'assets/placeholder.png'; ?>"
                                                 alt="Logo" class="img-thumbnail" width="60" height="60" style="object-fit: contain;"
                                                 onerror="this.onerror=null; this.src='assets/placeholder.png';">
                                        </td>
                                        <td class="align-middle"><?php echo htmlspecialchars($nombre_completo); ?></td>
                                        <td class="align-middle"><?php echo htmlspecialchars($resultado['partido_politico'] ?: 'Independiente'); ?></td>
                                        <td class="text-end align-middle fw-medium"><?php echo number_format($resultado['total_votos']); ?></td>
                                        <td class="text-end align-middle">
                                            <div class="progress" style="height: 25px;" title="<?php echo number_format($porcentaje, 2); ?>%">
                                                <div class="progress-bar bg-info text-dark" role="progressbar" style="width: <?php echo number_format($porcentaje, 2); ?>%;"
                                                     aria-valuenow="<?php echo number_format($porcentaje, 2); ?>" aria-valuemin="0" aria-valuemax="100">
                                                    <strong><?php echo number_format($porcentaje, 2); ?>%</strong>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                            <tfoot class="table-light fw-bold">
                                 <tr>
                                     <td colspan="3" class="text-end">TOTAL</td>
                                     <td class="text-end"><?php echo number_format($total_general_votos); ?></td>
                                     <td class="text-end">100.00%</td>
                                 </tr>
                             </tfoot>
                        </table>
                    </div>
                </div>
            </div>
        <?php endif; ?>
    </main>
    <footer class="text-center footer-info"> Sistema de Votaciones © <?php echo date('Y'); ?>. </footer>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
<?php if (!empty($chart_data)): ?>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function () {
        const canvas = document.getElementById('resultsChart');
        if(canvas) {
            const ctx = canvas.getContext('2d');
            const labels = <?php echo json_encode($chart_labels); ?>;
            const dataValues = <?php echo json_encode($chart_data); ?>;
            const backgroundColors = <?php echo json_encode($chart_background_colors); ?>;
            new Chart(ctx, {
                type: 'bar',
                data: { labels: labels, datasets: [{ label: 'Total Votos', data: dataValues, backgroundColor: backgroundColors, borderWidth: 1 }] },
                options: {
                    indexAxis: 'y', // Barras horizontales
                    responsive: true, maintainAspectRatio: false,
                    plugins: { legend: { display: false }, title: { display: true, text: 'Distribución de Votos' } },
                    scales: { y: { beginAtZero: true, ticks: { display: false } }, x: { title: { display: true, text: 'Número de Votos' }, beginAtZero: true } }
                }
            });
        }
    });
</script>
<?php endif; ?>
</body>
</html>