<?php
// *** PROCESAMIENTO PHP ***
// Incluido por index.php, $pdo y $_SESSION disponibles.

$mensaje_exito = '';
$mensaje_error = '';
$id_eleccion_actual = DEFAULT_ELECTION_ID;
$mesa_a_editar = null; // Para guardar datos en modo edición
$modo_edicion = false; // Flag

// --- Mensajes de éxito/error por GET ---
if (isset($_GET['success'])) {
    switch ($_GET['success']) {
        case 'created': $mensaje_exito = "¡Mesa registrada exitosamente!"; break;
        case 'updated': $mensaje_exito = "¡Número de Mesa actualizado exitosamente!"; break;
        case 'deleted': $mensaje_exito = "¡Mesa eliminada exitosamente!"; break;
    }
}
if (isset($_GET['error'])) {
     $mensaje_error = "Ocurrió un error: " . htmlspecialchars(urldecode($_GET['error']));
}

// --- DETECTAR MODO EDICIÓN (CARGAR DATOS) ---
if (isset($_GET['edit_id'])) {
    $id_para_editar = filter_input(INPUT_GET, 'edit_id', FILTER_VALIDATE_INT);
    if ($id_para_editar) {
        try {
            // Cargar datos de la mesa Y del centro asociado
            $sql_edit = "SELECT cv.id, cv.id_centro, cv.numero_mesa, c.localidad, c.nombre_ie
                         FROM centros_votacion cv
                         JOIN centros c ON cv.id_centro = c.id
                         WHERE cv.id = ? AND cv.id_eleccion = ?";
            $q_edit = $pdo->prepare($sql_edit);
            $q_edit->execute([$id_para_editar, $id_eleccion_actual]);
            $mesa_a_editar = $q_edit->fetch(PDO::FETCH_ASSOC);

            if ($mesa_a_editar) {
                $modo_edicion = true;
            } else {
                header("Location: index.php?page=centros_votacion&error=" . urlencode("Mesa no encontrada o no pertenece a esta elección."));
                exit;
            }
        } catch (PDOException $e) {
            error_log("Error al cargar mesa para editar: " . $e->getMessage());
            $mensaje_error = "Error al cargar datos de la mesa para edición.";
        }
    } else {
         header("Location: index.php?page=centros_votacion&error=" . urlencode("ID de mesa inválido para editar."));
         exit;
    }
}


// --- PROCESAR ELIMINACIÓN ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_id']) && isset($_POST['delete_type']) && $_POST['delete_type'] === 'mesa') {
    $id_para_eliminar = filter_input(INPUT_POST, 'delete_id', FILTER_VALIDATE_INT);
    $error_url_param = '';

    if ($id_para_eliminar) {
        try {
             $pdo->beginTransaction();

             // 1. Verificar si hay votos asociados a esta mesa específica (id en centros_votacion)
             $sql_check_votos = "SELECT COUNT(*) FROM votos WHERE id_centro_votacion = ?";
             $q_check = $pdo->prepare($sql_check_votos);
             $q_check->execute([$id_para_eliminar]);
             if ($q_check->fetchColumn() > 0) {
                 throw new Exception("No se puede eliminar la mesa porque tiene votos registrados.");
             }

            // 2. Eliminar de la tabla centros_votacion
            $sql_delete = "DELETE FROM centros_votacion WHERE id = ? AND id_eleccion = ?";
            $q_delete = $pdo->prepare($sql_delete);
            $q_delete->execute([$id_para_eliminar, $id_eleccion_actual]);

            if ($q_delete->rowCount() > 0) {
                $pdo->commit();
                header("Location: index.php?page=centros_votacion&success=deleted");
                exit;
            } else {
                 $pdo->rollBack();
                 $error_url_param = urlencode("No se encontró la mesa para eliminar (ID: $id_para_eliminar).");
            }
        } catch (PDOException $e) {
             $pdo->rollBack();
             error_log("Error PDO al eliminar mesa: " . $e->getMessage());
             $error_url_param = urlencode("Error de base de datos al eliminar la mesa.");
        } catch (Exception $e) { // Capturar excepción de votos
            $pdo->rollBack();
            error_log("Error al eliminar mesa: " . $e->getMessage());
            $error_url_param = urlencode($e->getMessage());
        }
    } else {
        $error_url_param = urlencode("ID inválido para eliminar.");
    }
    header("Location: index.php?page=centros_votacion&error=" . $error_url_param);
    exit;
}


// --- Procesar Formulario (Crear Mesa o Actualizar Número de Mesa) ---
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['guardar_centro_mesa'])) {
    $id_mesa_form = filter_input(INPUT_POST, 'id_mesa', FILTER_VALIDATE_INT); // ID si es actualización
    $localidad = trim($_POST['localidad_ie'] ?? '');
    $nombre_ie = trim($_POST['nombre_ie'] ?? '');
    $numero_mesa = trim($_POST['numero_mesa'] ?? '');

    // Validaciones
    if (empty($numero_mesa)) {
        $mensaje_error = "El número de mesa es obligatorio.";
    } elseif ($id_mesa_form === false && (empty($localidad) || empty($nombre_ie))) {
        // Localidad y Nombre IE solo son obligatorios al CREAR
        $mensaje_error = "La Localidad y el Nombre del IE son obligatorios al registrar una nueva mesa.";
    } else {
        try {
            if ($id_mesa_form) { // --- ACTUALIZAR (solo número de mesa) ---
                $sql_update = "UPDATE centros_votacion SET numero_mesa = ?, updated_at = NOW()
                               WHERE id = ? AND id_eleccion = ?";
                $q_update = $pdo->prepare($sql_update);
                // Comprobar si la ejecución fue exitosa o si el número no cambió
                if ($q_update->execute([$numero_mesa, $id_mesa_form, $id_eleccion_actual])) {
                     header("Location: index.php?page=centros_votacion&success=updated");
                     exit;
                } else {
                    // No debería fallar a menos que haya un problema grave, pero por si acaso
                    $mensaje_error = "No se pudo actualizar el número de mesa.";
                }

            } else { // --- CREAR (Lógica existente) ---
                $pdo->beginTransaction();
                // 1. Buscar o Crear el Centro Físico
                $id_centro = null;
                $sql_find_centro = "SELECT id FROM centros WHERE localidad = ? AND nombre_ie = ?";
                $q_find = $pdo->prepare($sql_find_centro);
                $q_find->execute([$localidad, $nombre_ie]);
                $centro_existente = $q_find->fetch(PDO::FETCH_ASSOC);

                if ($centro_existente) {
                    $id_centro = $centro_existente['id'];
                } else {
                    $sql_insert_centro = "INSERT INTO centros (localidad, nombre_ie, created_at, updated_at) VALUES (?, ?, NOW(), NOW())";
                    $q_insert_centro = $pdo->prepare($sql_insert_centro);
                    $q_insert_centro->execute([$localidad, $nombre_ie]);
                    $id_centro = $pdo->lastInsertId();
                }

                // 2. Crear la Mesa en centros_votacion
                if ($id_centro) {
                    $sql_insert_mesa = "INSERT INTO centros_votacion (id_eleccion, id_centro, numero_mesa, created_at, updated_at)
                                        VALUES (?, ?, ?, NOW(), NOW())";
                    $q_insert_mesa = $pdo->prepare($sql_insert_mesa);
                    $q_insert_mesa->execute([$id_eleccion_actual, $id_centro, $numero_mesa]);
                    $pdo->commit();
                    header("Location: index.php?page=centros_votacion&success=created");
                    exit;
                } else {
                     $pdo->rollBack();
                     $mensaje_error = "Error al obtener o crear el ID del centro físico.";
                }
            }
        } catch (PDOException $e) {
            if ($pdo->inTransaction()) { $pdo->rollBack(); } // Asegurar rollback si falla
            error_log("Error DB centro/mesa: " . $e->getMessage());
            if ($e->getCode() == '23000') {
                 $mensaje_error = "Error: La mesa '$numero_mesa' ya existe en este centro para esta elección.";
            } else {
                 $mensaje_error = "Error de base de datos al " . ($id_mesa_form ? 'actualizar' : 'guardar') . " la mesa.";
            }
             // Recargar datos si estábamos editando para no perderlos visualmente
            if ($id_mesa_form) { $modo_edicion = true; $mesa_a_editar = $_POST; $mesa_a_editar['id'] = $id_mesa_form; }
        }
    }
     // Si hubo error de validación, asegurar que seguimos en modo edición si aplica
    if (!empty($mensaje_error) && $id_mesa_form) {
         $modo_edicion = true;
         // Recargar datos originales para los campos readonly
         $sql_reload = "SELECT cv.id, cv.id_centro, cv.numero_mesa, c.localidad, c.nombre_ie FROM centros_votacion cv JOIN centros c ON cv.id_centro = c.id WHERE cv.id = ?";
         $q_reload = $pdo->prepare($sql_reload); $q_reload->execute([$id_mesa_form]);
         $mesa_a_editar = $q_reload->fetch(PDO::FETCH_ASSOC);
         $mesa_a_editar['numero_mesa'] = $_POST['numero_mesa']; // Mantener el número de mesa que intentó guardar
    }
}


// --- Cargar Lista de Mesas (Leer) ---
$lista_mesas = [];
$error_carga_lista = null;
try {
    $sql_select = "SELECT cv.id, c.localidad, c.nombre_ie, cv.numero_mesa
                   FROM centros_votacion cv
                   JOIN centros c ON cv.id_centro = c.id
                   WHERE cv.id_eleccion = ?
                   ORDER BY c.localidad, c.nombre_ie, cv.numero_mesa";
    $q_select = $pdo->prepare($sql_select);
    $q_select->execute([$id_eleccion_actual]);
    $lista_mesas = $q_select->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    error_log("Error al cargar mesas: " . $e->getMessage());
    $error_carga_lista = "Error al cargar la lista de centros/mesas.";
}

?>

<div class="container-fluid">
    <h1 class="mb-4"><i class="fas fa-map-marker-alt"></i> Gestión de Centros y Mesas (Elección ID: <?php echo $id_eleccion_actual; ?>)</h1>

    <!-- Área de mensajes -->
    <?php if ($mensaje_exito): ?><div class="alert alert-success alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_exito); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($mensaje_error): ?><div class="alert alert-danger alert-dismissible fade show" role="alert"><?php echo htmlspecialchars($mensaje_error); ?><button type="button" class="btn-close" data-bs-dismiss="alert"></button></div><?php endif; ?>
    <?php if ($error_carga_lista): ?><div class="alert alert-warning"><?php echo htmlspecialchars($error_carga_lista); ?></div><?php endif; ?>

    <!-- Formulario para agregar/editar -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
             <h6 class="m-0 font-weight-bold text-primary"><?php echo $modo_edicion ? 'Editar Número de Mesa' : 'Registrar Nueva Mesa'; ?></h6>
        </div>
        <div class="card-body">
            <form action="index.php?page=centros_votacion<?php echo $modo_edicion ? '&edit_id='.htmlspecialchars($mesa_a_editar['id']) : ''; ?>" method="POST">
                <?php if ($modo_edicion): ?>
                    <input type="hidden" name="id_mesa" value="<?php echo htmlspecialchars($mesa_a_editar['id']); ?>">
                <?php endif; ?>

                <div class="row mb-3">
                    <div class="col-md-6">
                        <label for="localidad_ie" class="form-label">Localidad del IE: <span class="text-danger <?php echo $modo_edicion ? 'd-none' : '';?>">*</span></label>
                        <input type="text" class="form-control" id="localidad_ie" name="localidad_ie" placeholder="Ej: Distrito Capital, Parroquia Sucre"
                               value="<?php echo htmlspecialchars($mesa_a_editar['localidad'] ?? ($_POST['localidad_ie'] ?? '')); ?>"
                               <?php echo $modo_edicion ? 'readonly' : 'required'; // Solo lectura en modo edición ?>>
                               <?php if ($modo_edicion): ?> <small class="form-text text-muted">No se puede cambiar la localidad al editar una mesa.</small> <?php endif; ?>
                    </div>
                    <div class="col-md-6">
                        <label for="nombre_ie" class="form-label">Nombre del IE: <span class="text-danger <?php echo $modo_edicion ? 'd-none' : '';?>">*</span></label>
                        <input type="text" class="form-control" id="nombre_ie" name="nombre_ie" placeholder="Ej: Escuela Básica Nal Bolivariana..."
                               value="<?php echo htmlspecialchars($mesa_a_editar['nombre_ie'] ?? ($_POST['nombre_ie'] ?? '')); ?>"
                               <?php echo $modo_edicion ? 'readonly' : 'required'; // Solo lectura en modo edición ?>>
                               <?php if ($modo_edicion): ?> <small class="form-text text-muted">No se puede cambiar el centro al editar una mesa.</small> <?php endif; ?>
                    </div>
                </div>
                <div class="row mb-3">
                     <div class="col-md-6">
                        <label for="numero_mesa" class="form-label">Nº de Mesa de Votación: <span class="text-danger">*</span></label>
                        <input type="text" class="form-control <?php echo (!empty($mensaje_error) && empty($_POST['numero_mesa'] ?? ($mesa_a_editar['numero_mesa'] ?? '') )) ? 'is-invalid' : ''; ?>"
                               id="numero_mesa" name="numero_mesa" placeholder="Ej: 01, 15B, M25" required
                               value="<?php echo htmlspecialchars($mesa_a_editar['numero_mesa'] ?? ($_POST['numero_mesa'] ?? '')); ?>">
                         <small class="form-text text-muted">Identificador único de la mesa dentro del centro para esta elección.</small>
                    </div>
                </div>
                <input type="hidden" name="guardar_centro_mesa" value="1">

                <button type="submit" class="btn btn-primary">
                    <i class="fas fa-<?php echo $modo_edicion ? 'sync-alt' : 'save'; ?>"></i> <?php echo $modo_edicion ? 'Actualizar Número' : 'Guardar Mesa'; ?>
                </button>
                 <?php if ($modo_edicion): ?>
                <a href="index.php?page=centros_votacion" class="btn btn-secondary">
                    <i class="fas fa-times"></i> Cancelar Edición
                </a>
                <?php else: ?>
                <button type="reset" class="btn btn-secondary"><i class="fas fa-eraser"></i> Limpiar</button>
                <?php endif; ?>
            </form>
        </div>
    </div>

    <!-- Tabla para listar mesas -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Lista de Mesas Registradas</h6>
        </div>
        <div class="card-body">
             <?php if (empty($lista_mesas) && empty($error_carga_lista)): ?>
                <p class="text-center">No hay mesas registradas para esta elección.</p>
            <?php elseif (!empty($lista_mesas)): ?>
                 <div class="table-responsive">
                    <table class="table table-bordered table-hover table-sm" id="dataTableCentros" width="100%" cellspacing="0">
                        <thead>
                            <tr>
                                <th>ID Mesa</th>
                                <th>Localidad IE</th>
                                <th>Nombre IE</th>
                                <th>Nº Mesa</th>
                                <th class="text-center">Acciones</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($lista_mesas as $mesa):
                                $nombre_descriptivo_mesa = "Mesa " . htmlspecialchars($mesa['numero_mesa']) . " en " . htmlspecialchars($mesa['nombre_ie']);
                            ?>
                            <tr>
                                <td><?php echo htmlspecialchars($mesa['id']); ?></td>
                                <td><?php echo htmlspecialchars($mesa['localidad']); ?></td>
                                <td><?php echo htmlspecialchars($mesa['nombre_ie']); ?></td>
                                <td><?php echo htmlspecialchars($mesa['numero_mesa']); ?></td>
                                <td class="text-center align-middle">
                                     <a href="index.php?page=centros_votacion&edit_id=<?php echo htmlspecialchars($mesa['id']); ?>" class="btn btn-sm btn-warning me-1" title="Editar Número de Mesa">
                                            <i class="fas fa-edit"></i>
                                     </a>
                                     <button type="button" class="btn btn-sm btn-danger" title="Eliminar Mesa"
                                                data-bs-toggle="modal" data-bs-target="#confirmDeleteModal"
                                                data-id="<?php echo htmlspecialchars($mesa['id']); ?>"
                                                data-name="<?php echo $nombre_descriptivo_mesa; ?>"
                                                data-type="mesa"
                                                data-page="centros_votacion">
                                            <i class="fas fa-trash"></i>
                                      </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
             <?php endif; ?>
        </div>
    </div>
</div> <!-- Fin container-fluid -->